/*******************************************************************************
 * Copyright (c) 2000, 2015 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ant.tests.ui.editor.performance;

import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.File;

import org.eclipse.ant.internal.ui.AntUIPlugin;
import org.eclipse.ant.internal.ui.preferences.AntEditorPreferenceConstants;
import org.eclipse.ant.tests.ui.performance.AbstractAntPerformanceTest;
import org.eclipse.ant.tests.ui.testplugin.ProjectHelper;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.test.performance.Dimension;
import org.eclipse.ui.PartInitException;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;

@SuppressWarnings("restriction")
public class OpenAntEditorTest extends AbstractAntPerformanceTest {

	@BeforeEach
	@Override
	public void setUp(TestInfo testInfo) throws Exception {
		super.setUp(testInfo);
		EditorTestHelper.runEventQueue();
	}

	@Test
	public void testOpenAntEditor1() throws Exception {
		// cold run
		IFile file = getIFile("build.xml"); //$NON-NLS-1$
		measureOpenInEditor(file);
	}

	@Test
	public void testOpenAntEditor2() throws Exception {
		// warm run
		IFile file = getIFile("build.xml"); //$NON-NLS-1$
		tagAsGlobalSummary("Open Ant Editor", Dimension.ELAPSED_PROCESS); //$NON-NLS-1$
		measureOpenInEditor(file);
	}

	@Test
	public void testOpenAntEditorNoFolding() throws Exception {
		IPreferenceStore store = AntUIPlugin.getDefault().getPreferenceStore();
		try {
			IFile file = getIFile("build.xml"); //$NON-NLS-1$
			store.setValue(AntEditorPreferenceConstants.EDITOR_FOLDING_ENABLED, false);
			tagAsSummary("Open Ant Editor; No folding", Dimension.ELAPSED_PROCESS); //$NON-NLS-1$
			measureOpenInEditor(file);
		}
		finally {
			store.setToDefault(AntEditorPreferenceConstants.EDITOR_FOLDING_ENABLED);
		}
	}

	protected IFile getIFile(String buildFileName) {
		return getProject().getFolder("buildfiles").getFolder("performance").getFile(buildFileName); //$NON-NLS-1$ //$NON-NLS-2$
	}

	protected File getBuildFile(String buildFileName) {
		IFile file = getIFile(buildFileName);
		assertTrue(file.exists(), "Could not find build file named: " + buildFileName); //$NON-NLS-1$
		return file.getLocation().toFile();
	}

	/**
	 * Returns the 'AntUITests' project.
	 *
	 * @return the test project
	 */
	protected IProject getProject() {
		return ResourcesPlugin.getWorkspace().getRoot().getProject(ProjectHelper.PROJECT_NAME);
	}

	synchronized protected void measureOpenInEditor(IFile file) throws PartInitException, InterruptedException {
		try {
			for (int i = 0; i < 15; i++) {
				startMeasuring();
				EditorTestHelper.openInEditor(file, true);
				stopMeasuring();
				EditorTestHelper.closeAllEditors();
				wait(2000); // NOTE: runnables posted from other threads, while the main thread waits here, are executed and measured only in the next
				// iteration
			}
			commitMeasurements();
			assertPerformance();
		}
		finally {
			EditorTestHelper.closeAllEditors();
		}
	}
}
